//*************************************************************************************************
//
//	Description:
//		wheels.fx - Special purpose shader for rendering translucent wheel hubs with blending between
//		two sets of textures with different blur levels.
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     06/15/2007  0.1           Created
//		TNettleship			07/11/2007	0.2						Changed lighting to work in world coords.
//		TNettleship     07/24/2007  0.3	          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  0.4						Removed texture bias.
//		TNettleship     10/10/2007  0.5						Added specialisations for fresnel and chrome effect.
//		TNettleship			10/23/2007	0.6						Converted to on-load rendermode behaviour binding.
//		TNettleship			10/26/2007	0.7						Removed blurred normal map.
//	<TABLE>
//
//*************************************************************************************************
#define _SSAO_READY_
#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define USE_FRESNEL
#define CHROME_EFFECT
#define ENV_MAPPING
#define USE_LIGHT_CONTROL
#endif

#define MAX_ENVMAP_BIAS	6.0f

// Uncomment to use experimental higher quality envmapping approach
#define EXPERIMENTAL_ENVMAPPING


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#elif defined( ENV_MAPPING )
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
	bool use3dMapping = true;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Used for the emissive lighting control
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool ProtectFromRescaling = true;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Simulation constants
//

float angularVelocity													// Angular velocity of the wheel (in rad/s)
<
	string UIName = "Angular velocity";
	float UIMin = -1024.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 0.0f;

float minBlurVelocity														// Angular velocity of minimum blur map
<
	string UIName = "Min Blur Vel";
	bool appEdit = true;
> = 0.0f;

float maxBlurVelocity														// Angular velocity of maximum blur map
<
	string UIName = "Max Blur Vel";
	bool appEdit = true;
> = 1.0f;

//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Spec Tex {UV1}";
	bool appEdit = true;
>;

texture normalTexture : TEXTURE							// Normal map
<
	string UIName = "Normal Tex {UV1}";
	bool appEdit = true;
>;

texture blurredDiffuseTexture : TEXTURE							
<
	string UIName = "Blurred Diff {UV1}";
	bool appEdit = true;
>;

texture blurredSpecularTexture : TEXTURE							
<
	string UIName = "Blurred Spec {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useFresnel, "Use Fresnel?", "USE_FRESNEL" )	// TRUE if fresnel factor should be used

#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fresnelFactor, "Fresnel Factor", useFresnel, 0.0f, 1.0f, 0.6f )
#endif

SPECIALISATION_PARAM( useChromeEffect, "Chrome effect?", "CHROME_EFFECT" )	// TRUE if chrome effect is to be used
#if defined( _3DSMAX_ ) || defined( CHROME_EFFECT )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( chromeColour, "Chrome Colour", useChromeEffect, 1.0f, 1.0f, 1.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_DEFAULTED( chromePower, "Chrome Power", useChromeEffect, 8.0f )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 65536.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 65536.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
> = 1.0f;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 16.0f, 1.0f )
#endif

// TRUE if the material represents vehicle lights & so self-emits under code control
SPECIALISATION_PARAM( useLightControl, "Light Control?", "USE_LIGHT_CONTROL" )
DECLARE_DEPENDENT_VERTEX_STREAM( texcoord2Dependency, texCoord2, TEXCOORD2, useLightControl )

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
DEPENDENT_TEXTURE_PARAM( emissiveTexture, "Emissive Tex{UV1}", useLightControl )

#define NUM_EMISSION_CONTROL_VALUES		20
float emissionControl[ NUM_EMISSION_CONTROL_VALUES ]
<
	string UIName = "emissionControl";
	bool appEdit = true;
	bool export = false;
	string exportDependsOn = "useLightControl";
>;

float testLightLevel
<
	string UIName = "Test Light Level";
	bool appEdit = false;
	bool export = false;
> = 0.0f;
#endif


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


sampler2D blurredDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="blurredDiffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
  Texture = < blurredDiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
  MinFilter = _MINFILTER;
  MagFilter = Linear;
  MipFilter = Linear;
  AddressU  = Wrap;
  AddressV  = Wrap;
#if defined(_PS3_)
  LODBias = 0;
#else
  MipMapLODBias = 0;
#endif
  SET_NO_ANISOTROPY
 #endif
};

sampler2D blurredSpecularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="blurredSpecularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
  Texture = < blurredSpecularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
  MinFilter = _MINFILTER;
  MagFilter = Linear;
  MipFilter = Linear;
  AddressU  = Wrap;
  AddressV  = Wrap;
#if defined(_PS3_)
  LODBias = 0;
#else
  MipMapLODBias = 0;
#endif
  SET_NO_ANISOTROPY
 #endif
};

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined ( USE_LIGHT_CONTROL )
sampler2D emissiveMap : SAMPLER 
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="emissiveTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < emissiveTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


//
// Functions
//


//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for car paint is 0.6667 (which assumes air = 1.0, clearcoat = 1.5)
//
float calculated_fresnel(float3 _eye, float3 _normal, float rindexRatio)
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow( 1.0f - rindexRatio, 2.0f ) / pow ( 1.0f + rindexRatio, 2.0f );

	return saturate( R0 + ( 1.0f - R0 ) * pow(1.0f - saturate( dot( -_eye, _normal ) ), 5.0f ) );
}



// Function calculates emissiveness by looking up a value from an array which the app sets,
// indexed by the control ID passed. If a negative ID is passed, the emission is considered to
// be always on.
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
float CalcEmissiveness( float _ID, float _brightness )
{
	int intControlID = floor( _ID );

	if ( intControlID >= 0 )
	{
#if defined( _3DSMAX_ )
		return testLightLevel * abs( _brightness );
#else
		return emissionControl[ intControlID ] * abs( _brightness );
#endif
	}
	else
	{
		return 0.0f;
	}
}
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float3 texCoord : TEXCOORD0;													// UVW channel 1 texture coord - N.B. Max requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float2 texCoord2 : TEXCOORD2;													// UV channel 2 texture coord (emissive control)
#else
	float4 colour   : COLOR0;															// Vertex colour
	float3 texCoord : TEXCOORD0;													// UVW channel 1 texture coord

#if defined( USE_LIGHT_CONTROL )
	float2 texCoord2 : TEXCOORD1;													// UV channel 2 texture coord (emissive control)
#endif

#endif

	float3 normal   : NORMAL;															// Object space normal
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD5;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 normal_plus_fresnel	: TEXCOORD1;							// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	float3 tangent		: TEXCOORD3;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD4;												// Binormal vector (world space)

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float emissiveness : TEXCOORD6;												// Emissiveness amount
#endif

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
};



struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
	// This is only needed if we're doing envmapping
	float4 reflectionVector_plus_fresnel	: TEXCOORD1;		// Reflection vector (envmap space), with fresnel term in w
#endif
#if defined( _3DSMAX_ ) || defined( CHROME_EFFECT )
	// This is only needed if we're doing the chrome effect
	float metallicFactor	: TEXCOORD2;										// Metallic chrome effect factor
#endif
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float emissiveness : TEXCOORD3;										// Emissiveness amount
#endif
};


struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord			: TEXCOORD0;
#ifdef NEEDS_SHADOW_COORDS
	float4 shadowCoord	: TEXCOORD1;
#endif
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT WheelsVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal		 = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
	_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
		_output.emissiveness = CalcEmissiveness( _input.texCoord2.x, _input.texCoord2.y );
#endif
	DEPENDENT_CODE_END( useLightControl )

	// If its required
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		// Calculate fresnel term
		_output.normal_plus_fresnel = float4( normal, calculated_fresnel( - normalize( worldEyeVec ), normal, fresnelFactor ) );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		// Fresnel term is 1.0
		_output.normal_plus_fresnel = float4( normal, 1.0f );
#endif
	DEPENDENT_CODE_END( useFresnel )

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}


VSOUTPUT_LD WheelsLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoord = _input.texCoord;

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	float3 eye = normalize( worldEyeVec );

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
		_output.emissiveness = CalcEmissiveness( _input.texCoord2.x, _input.texCoord2.y );
#endif
	DEPENDENT_CODE_END( useLightControl )

	// If chrome effect is required
	DEPENDENT_CODE_START( useChromeEffect )
#if defined( _3DSMAX_ ) || defined( CHROME_EFFECT )
		// Calculate chrome factor
		float ndotv = saturate( dot( normal, eye ) );
		_output.metallicFactor = pow( ndotv, chromePower );
#endif
	DEPENDENT_CODE_END( useChromeEffect )

	// If envmapping is required
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )

	// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		float3 envMapSpaceNormal = mul( float4( normal.xyz, 0.0f ), worldI ).xyz;
		float3 envMapSpaceEye = mul( -eye, worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		float3 envMapSpaceNormal = mul( float4( normal.xyz, 0.0f ), envTransformI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif
		float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

		// If its required
		DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
			// Calculate fresnel term
			_output.reflectionVector_plus_fresnel = float4( reflectionVector, calculated_fresnel( -eye, normal, fresnelFactor ) );
#endif
		DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
			// Fresnel term is 1.0
			_output.reflectionVector_plus_fresnel = float4( reflectionVector, 1.0f );
#endif
		DEPENDENT_CODE_END( useFresnel )

#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Calculate vertex lighting (diffuse + ambient)
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
	_output.colour.a = _input.colour.a;

	return _output;
}



SHADOWGEN_VSOUTPUT WheelsShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
#ifdef NEEDS_SHADOW_COORDS
	OUTPUT_SHADOW_COORDS;
#endif	

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD5;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 normal_plus_fresnel	: TEXCOORD1;							// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD3;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD4;												// Binormal vector (world space)
	float emissiveness : TEXCOORD6;												// Emissiveness amount

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD5;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 normal_plus_fresnel	: TEXCOORD1_centroid;			// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)

	float3 tangent		: TEXCOORD3_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD4_centroid;								// Binormal vector (world space)

#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// This vertex stream isn't needed without emissive lighting
	float emissiveness : TEXCOORD6_centroid;				// Emissiveness amount
#endif

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif


struct SHADOWGEN_PSINPUT
{
	float2 texCoord : TEXCOORD0;
#ifdef NEEDS_SHADOW_COORDS
	float4 shadowCoord	: TEXCOORD1;
#endif	
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

#ifdef _POST_PARSE_
PSOUTPUT WheelsFragmentShader( PSINPUT _input )
#else
PSOUTPUT WheelsFragmentShader( PSINPUT _input, uniform bool _opaque )
#endif
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;

	// Read textures
	float4 unblurredDiffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 unblurredSpecularTexColour = tex2D( specularMap, _input.texCoord );
	float4 blurredDiffuseTexColour = tex2D( blurredDiffuseMap, _input.texCoord );
	float4 blurredSpecularTexColour = tex2D( blurredSpecularMap, _input.texCoord );
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord );
#endif

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal_plus_fresnel.xyz );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Normalise the input tangent and binormal vectors
	float3 tangent = normalize( _input.tangent );
	float3 binormal = normalize( _input.binormal );

	// Fetch and decode the map normals
	float4 normalMapColour = tex2D( normalMap, _input.texCoord );
	float3 normalFromMap = ( normalMapColour.rgb * 2.0f ) - 1.0f;

	// Interpolate between the unblurred and blurred diffuse colours and normals
	float interpolant = clamp( abs( angularVelocity ), minBlurVelocity, maxBlurVelocity );
	interpolant = ( interpolant - minBlurVelocity ) / ( maxBlurVelocity - minBlurVelocity );

	float4 diffuseTexColour = lerp( unblurredDiffuseTexColour, blurredDiffuseTexColour, interpolant );
	float4 specularTexColour = lerp( unblurredSpecularTexColour, blurredSpecularTexColour, interpolant );

	// If this is a genuinely opaque material
	if ( _opaque )
	{
		// Force the diffuse alpha to 1.0 (this works around problems with destination alpha = 0.0 on
		// wheel materials which the artists have doubled diffuse and specular maps)
		diffuseTexColour.a = 1.0f;
	}

	// Perturb the tangent space normal by the normal map
	normal = ( normalFromMap.z * TSnormal ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
	normal = lerp( normal, TSnormal, interpolant );
	normal = normalize( normal );

	//
	// Chrome effect
	//

	DEPENDENT_CODE_START( useChromeEffect )
#if defined( _3DSMAX_ ) || defined( CHROME_EFFECT )
		float ndotv = saturate( dot( normal, eye ) );

		// Modify base colour with metallic accent colour
		diffuseTexColour.rgb += chromeColour.rgb * pow( ndotv, chromePower );
#endif
	DEPENDENT_CODE_END( useChromeEffect )

	//
	// Calculate the fresnel factor
	//
	float fresnelCoefficient;

	// Basic coefficient
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		fresnelCoefficient = saturate( _input.normal_plus_fresnel.w );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		fresnelCoefficient = 1.0f;
#endif
	DEPENDENT_CODE_END( useFresnel )


#if defined( PSEUDO_GI )

	// If environment mapping is switched on
	float4 accumulatedColour;
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate ambient-lit base colour (looking up the envmap for the ambient lighting)
		float4 envmapambient = texCUBElod( environmentMap, float4( normal.xyz, 4 ) );
		accumulatedColour = diffuseTexColour * envmapambient;
#endif
	DEPENDENT_CODE_ELSE( useEnvironmentMap )
#if defined( _3DSMAX_ ) || !defined( ENV_MAPPING )
		// Calculate ambient-lit base colour from interpolated lighting
		accumulatedColour = diffuseTexColour * _input.colour;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

#else
	float4 accumulatedColour = diffuseTexColour * _input.colour;
#endif

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// Add in emissive lighting
	accumulatedColour.rgb += _input.emissiveness * emissiveColour.rgb * emissiveColour.a;
#endif
	DEPENDENT_CODE_END( useLightControl )

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), envTransformI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif
		float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

#if defined( EXPERIMENTAL_ENVMAPPING )
	// Intersect the reflection vector with a 4m sphere centred on the car
	float3 envMapSpacePos = mul( float4( worldPos, 1.0f ), envTransformI ).xyz;
	float3 rayOrigin = envMapSpacePos;
	float3 rayDirection = reflectionVector;
	float B = dot( rayDirection, rayOrigin ) * 2.0f;
	float C = dot( rayOrigin, rayOrigin ) - ( 4.0f * 4.0f );
	float t0 = -B - sqrt( B*B - 4.0f * C ) / 2.0f;
	float t1 = -B + sqrt( B*B - 4.0f * C ) / 2.0f;
	float3 sphereIntersection = rayOrigin + ( rayDirection * max( t0, t1 ) );

	// Use the intersection point to look up the cubemap, not the reflection vector
	reflectionVector = normalize( sphereIntersection );
#endif

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour (back in tangent space is fine) and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor * fresnelCoefficient;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w * _input.colour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



PSOUTPUT WheelsLowDetailFragmentShader( VSOUTPUT_LD _input )
{
  PSOUTPUT _output;

  float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Chrome effect
	DEPENDENT_CODE_START( useChromeEffect )
#if defined( _3DSMAX_ ) || defined( CHROME_EFFECT )
		diffuseTexColour.rgb += chromeColour.rgb * _input.metallicFactor;
#endif
	DEPENDENT_CODE_END( useChromeEffect )

	// Apply vertex lighting
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	DEPENDENT_CODE_START( useLightControl )
#if defined( _3DSMAX_ ) || defined( USE_LIGHT_CONTROL )
	// Add in emissive lighting
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord );
	accumulatedColour.rgb += _input.emissiveness * emissiveColour.rgb * emissiveColour.a;
#endif
	DEPENDENT_CODE_END( useLightControl )

	// Apply envmapping
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )

#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBE( environmentMap, _input.reflectionVector_plus_fresnel.xzy );
#else
		float4 environmentTexColour = texCUBE( environmentMap, _input.reflectionVector_plus_fresnel.xyz );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += environmentTexColour * specularTexColour * globalEMapFactor * _input.reflectionVector_plus_fresnel.w;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}



PSOUTPUT WheelsShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT output;

	float4 unblurredDiffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 blurredDiffuseTexColour = tex2D( blurredDiffuseMap, _input.texCoord );

	float interpolant = clamp( abs( angularVelocity ), minBlurVelocity, maxBlurVelocity );
	interpolant = ( interpolant - minBlurVelocity ) / ( maxBlurVelocity - minBlurVelocity );

#ifdef NEEDS_SHADOW_COORDS
	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
#else
	output.Colour.rgb = 0;
#endif
	output.Colour.a = lerp( unblurredDiffuseTexColour.a, blurredDiffuseTexColour.a, interpolant );
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.5f );

	return output;
}




//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Wheels
<
	bool supportsSpecialisedLighting = true;
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Wheels";
	int    normalDeferredID		= 2;

	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";

	string shadowGenBehaviour  = "ERMB_RENDER";
	string shadowGenTechnique	 = "_Wheels_ShadowGen";
	int    shadowGenDeferredID = 0;

	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Wheels_LowDetail";
	int    lowDetailDeferredID = 2;	// needs to be like this as due to a limitation in the engine, it'll be rendered using THIS techniques render states
	bool   usefullprecision = true;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx WheelsVertexShader();
		PixelShader = compile sce_fp_rsx WheelsFragmentShader( false );
#else
		VertexShader = compile vs_3_0 WheelsVertexShader();
		PixelShader = compile ps_3_0 WheelsFragmentShader( false );
#endif
	}
}




technique Wheels_Opaque
<
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Wheels_Opaque";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";

	string shadowGenBehaviour  = "ERMB_RENDER";
	string shadowGenTechnique	 = "_Wheels_ShadowGen";
	int    shadowGenDeferredID = 0;

	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Wheels_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   usefullprecision = true;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx WheelsVertexShader();
		PixelShader = compile sce_fp_rsx WheelsFragmentShader( true );
#else
		VertexShader = compile vs_3_0 WheelsVertexShader();
		PixelShader = compile ps_3_0 WheelsFragmentShader( true );
#endif
	}
}



technique Wheels_LowDetail
<
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Wheels_LowDetail";
	int    normalDeferredID		= 0;

	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";

	string shadowGenBehaviour  = "ERMB_RENDER";
	string shadowGenTechnique	 = "_Wheels_ShadowGen";
	int    shadowGenDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx WheelsLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx WheelsLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 WheelsLowDetailVertexShader();
		PixelShader = compile ps_3_0 WheelsLowDetailFragmentShader();
#endif
	}
}



technique _Wheels_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 0x40 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;

			VertexShader = compile sce_vp_rsx WheelsShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx WheelsShadowGenFragmentShader();
#elif defined(_XBOX)
			VertexShader = compile vs_3_0 WheelsShadowGenVertexShader();
			PixelShader = compile ps_3_0 WheelsShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;

			VertexShader = compile vs_3_0 WheelsShadowGenVertexShader();
			PixelShader = compile ps_3_0 WheelsShadowGenFragmentShader();
#endif
	}
}

